#!/bin/bash
# Red Giant Service runs as a dedicated user to limit its privileges.
# This script will create that user if necessary.

# Create directory for service to store files
echo "Creating service folder"
mkdir -p /Users/Shared/.service
sudo chmod -R a+w /Users/Shared/.service
sudo chmod -R a+r /Users/Shared/.service
sudo chmod -R a+r "/Library/Logs/Red Giant"
sudo chmod -R a+w "/Library/Logs/Red Giant"

sudo chmod -R a+w "/Users/Shared/Red Giant"
sudo chmod -R a+r "/Users/Shared/Red Giant"

##################################################################
# USER CREATION
# Our user name matches the service name
fuseUserName=_redgiantservice

# We check each individual part of the creation process in the hopes
# of repairing any pevious installs that might have been only
# partly successful, e.g., FUSE-1342

##################################################################
# Create the user if it wasn't already created on a previous install
existingUser=$(dscl . -list /Users | grep -Eo $fuseUserName)
if [ ! -z "$existingUser" ]; then
	echo "Service user $fuseUserName already exists"
else
	echo "Creating service user $fuseUserName"
	sudo dscl . -create /Users/$fuseUserName

	# Validate it actually worked
	createdUser=$(dscl . -list /Users | grep -Eo $fuseUserName)
	if [ -z "$createdUser" ]; then
		echo "Error: user not created"
		exit 1
	fi

fi

##################################################################
# Assign the user id if it doesn't already have one
existingUser=$(dscl . -list /Users UniqueID | grep -Eo $fuseUserName)
if [ ! -z "$existingUser" ]; then
	echo "Service user UniqueID already assigned"
else
	echo "Going to assign a UniqueID to the service user"

	# Get the next available user id
	highestUserId=$(dscl . -list /Users UniqueID | sort -nr -k 2 | head -1 | grep -Eo '[[:space:]][0-9]+' | grep -Eo '[0-9]+')
	if [ "$highestUserId" -lt "500" ]; then
		# UIDs from 0 through 500 are reserved for use by the system, so make sure we're above 500
		highestUserId=500
	fi
	newId=$((highestUserId + 1)) #one bigger than the highest on the system

	echo "Assigning ID $newId"
	sudo dscl . -create /Users/$fuseUserName UniqueID $newId

	# Validate it actually worked
	createdUser=$(dscl . -list /Users UniqueID | grep -Eo $fuseUserName)
	if [ -z "$createdUser" ]; then
		echo "Error: user UniqueID not assigned"
		exit 1
	fi
fi


##################################################################
# Assign the user shell if it doesn't already have one
existingUser=$(dscl . -list /Users UserShell | grep -Eo $fuseUserName)
if [ ! -z "$existingUser" ]; then
	echo "Service user shell already assigned"
else
	echo "Assigning a shell to the service user"

	sudo dscl . -create /Users/$fuseUserName UserShell /usr/bin/false

	# Validate it actually worked
	createdUser=$(dscl . -list /Users UserShell | grep -Eo $fuseUserName)
	if [ -z "$createdUser" ]; then
		echo "Error: user UserShell not assigned"
		exit 1
	fi
fi

##################################################################
# Create a group for our user if one doesn't exist
existingGroup=$(dscl . -list /Groups | grep -Eo $fuseUserName)
if [ ! -z "$existingGroup" ]; then
	echo "Service group $fuseUserName already exists"
else
	echo "Creating service group $fuseUserName"
	sudo dscl . -create /Groups/$fuseUserName

	# Do these too without a check since they aren't as critical
	sudo dscl . -create /Groups/$fuseUserName name $fuseUserName
	sudo dscl . -create /Groups/$fuseUserName passwd "*"

	# Validate it actually worked
	createdGroup=$(dscl . -list /Groups | grep -Eo $fuseUserName)
	if [ -z "$createdGroup" ]; then
		echo "Error: group not created"
		exit 1
	fi
fi


##################################################################
# Assign an ID to the group if it doesn't have one
fuseUserGroupId=$(dscl "/Local/Default" read /Groups/$fuseUserName PrimaryGroupID | grep -Eo '[0-9]+')

if [ ! -z "$fuseUserGroupId" ]; then
	echo "Service user group already has ID $fuseUserGroupId"
else
	echo "Assigning an ID to the user group"

	# Get the next available group id
	highestGroupId=$(dscl . -list /Groups PrimaryGroupID | sort -nr -k 2 | head -1 | grep -Eo '[[:space:]][0-9]+' | grep -Eo '[0-9]+')
	fuseUserGroupId=$((highestGroupId + 1)) #one bigger than the highest on the system

	echo "Assigning group id ID $fuseUserGroupId"
	sudo dscl . -create /Groups/$fuseUserName PrimaryGroupID $fuseUserGroupId

	# Validate that it actually worked
	createdGroupId=$(dscl "/Local/Default" read /Groups/$fuseUserName PrimaryGroupID | grep -Eo '[0-9]+')	
	if [ -z "$createdGroup" ]; then
		echo "Error: ID not assigned to group"
		exit 1
	fi
fi


##################################################################
# Assign the user group if it isn't already set to our group
existingUserGroupId=$(dscl . -list /Users PrimaryGroupID | grep $fuseUserName | grep -Eo '[0-9]+')
if [ "$existingUserGroupId" == "$fuseUserGroupId"  ]; then
	echo "Service user group already assigned"
else
	echo "Assigning a group to the service user"

	echo "Assigning group $fuseUserGroupId (was $existingUserGroupId)"
	sudo dscl . -create /Users/$fuseUserName PrimaryGroupID $fuseUserGroupId

	# Validate it actually worked
	createdUser=$(dscl . -list /Users PrimaryGroupID | grep -Eo $fuseUserName)
	if [ -z "$createdUser" ]; then
		echo "Error: user group not assigned"
		exit 1
	fi
fi

echo "Service user creation complete"
